#!/usr/bin/env python

# Copyright (C) 2007 Micah Carrick <email@micahcarrick.com>
# Copyright (C) 2007 ausland <ausland@ausland-berlin.de>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# -----------------------------------------------------------------------------
#



import sys
import os
import gtk
import pango
import ardour
import streaming

class RecordingGui:

    # When our window is destroyed, we want to break out of the GTK main loop. 
    # We do this by calling gtk_main_quit(). We could have also just specified 
    # gtk_main_quit as the handler in Glade!
    def on_window_destroy(self, widget, data=None):  
        gtk.main_quit()

    def on_window_delete_event(self, widget, data=None):
        return self.check_status_before_quit()
    
    # Called when the user clicks the 'Quit' menu. We need to prompt for save if 
    # the file has been modified and then break out of the GTK+ main loop          
    def on_quit_menu_item_activate(self, menuitem, data=None):
        if not self.check_status_before_quit():        
            gtk.main_quit()

    # Called when the user clicks the 'Start Darkice' menu. 
    # Checks first to see if darkice is already running          
    def on_darkice_start_menu_item_activate(self, menuitem, data=None):
        self.darkice_toggle_state()

    # Called when the user clicks the 'Stop Darkice' menu. 
    # Checks first to see if darkice is already running          
    def on_darkice_stop_menu_item_activate(self, menuitem, data=None):
        self.darkice_toggle_state()

    # Called when the user clicks the 'Start Darkice' menu. 
    # Checks first to see if darkice is already running          
    def on_darkice_togglebutton_clicked(self, button, data=None):
        self.darkice_toggle_state()

    # Called when the user clicks the 'Start Ardour' menu. 
    # Checks first to see if ardour is already running          
    def on_ardour_start_menu_item_activate(self, menuitem, data=None):
        self.ardour_start()

    # Called when the user clicks the 'externes darkice beenden' menu. 
    # Checks first to see if darkice is already running          
    def on_darkice_stop_external_menu_item_activate(self, menuitem, data=None):
        self.killall_darkice()

    # Called when the user clicks the 'Start Ardour' menu. 
    # Checks first to see if ardour is already running             
    def on_ardour_button_clicked(self, button, data=None):
        self.ardour_start()           
    
    # Called when the user clicks the 'About' menu. We use gtk_show_about_dialog() 
    # which is a convenience function to show a GtkAboutDialog. This dialog will
    # NOT be modal but will be on top of the main application window.    
    def on_about_menu_item_activate(self, menuitem, data=None):
    
        if self.about_dialog: 
            self.about_dialog.present()
            return
        
        authors = [
        "Micah Carrick <email@micahcarrick.com>",
        "Conrad Noack <ausland@ausland-berlin.de>"
        ]

        about_dialog = gtk.AboutDialog()
        about_dialog.set_transient_for(self.window)
        about_dialog.set_destroy_with_parent(True)
        about_dialog.set_name("Recording Gui")
        about_dialog.set_version("0.1")
        about_dialog.set_copyright("Copyright \xc2\xa9 2008 ausland")
        about_dialog.set_website("http://www.ausland-berlin.de")
        about_dialog.set_comments("starts and stops darkice with jack \nand starts ardour with a new session")
        about_dialog.set_authors            (authors)
        about_dialog.set_logo_icon_name     (gtk.STOCK_MEDIA_RECORD)
        
        # callbacks for destroying the dialog
        def close(dialog, response, editor):
            editor.about_dialog = None
            dialog.destroy()
            
        def delete_event(dialog, event, editor):
            editor.about_dialog = None
            return True
                    
        about_dialog.connect("response", close, self)
        about_dialog.connect("delete-event", delete_event, self)
        
        self.about_dialog = about_dialog
        about_dialog.show()

    # We call error_message() any time we want to display an error message to 
    # the user. It will both show an error dialog and log the error to the 
    # terminal window.
    def error_message(self, message):
    
        # log to terminal window
        print message
        
        # create an error message dialog and display modally to the user
        dialog = gtk.MessageDialog(None,
                                   gtk.DIALOG_MODAL | gtk.DIALOG_DESTROY_WITH_PARENT,
                                   gtk.MESSAGE_ERROR, gtk.BUTTONS_OK, message)
        
        dialog.run()
        dialog.destroy()

    def check_status_before_quit(self):
        if self.is_darkice_running():
            self.error_message('Bitte erst Streaming beenden, dann die Anwendung schliessen.')
            return True
        if self.is_ardour_running():
            self.error_message('Bitte erst Daten in Ardour speichern und Ardour beenden, dann die Anwendung schliessen.')
            return True

        return False

    def ardour_start(self):
        self.statusbar.push(self.statusbar_cid, 'Ardour starten...')
        self.darkice_stop_menu_item.set_sensitive(False) 
        self.ardour_button.set_sensitive(False)
        self.ardour_start_menu_item.set_sensitive(False)
        if self.start_ardour():
            self.statusbar.push(self.statusbar_cid, 'Ardour gestartet')

    def darkice_toggle_state(self):
        if not self.is_darkice_running():
            self.darkice_start_menu_item.set_sensitive(False)
            self.darkice_stop_menu_item.set_sensitive(True)
            #self.change_darkice_togglebutton_color(True)
            self.darkice_togglebutton.set_label('Streaming stoppen')            
            if self.start_darkice():
                self.statusbar.push(self.statusbar_cid, 'Streaming gestartet')
            else:
                self.disable_darkice()
        else: 
            self.statusbar.push(self.statusbar_cid, 'Streaming stoppen...')
            self.stop_darkice()
            self.darkice_stop_menu_item.set_sensitive(False)
            self.darkice_start_menu_item.set_sensitive(True)
            #self.change_darkice_togglebutton_color(False)
            self.darkice_togglebutton.set_label('Streaming starten')
            self.statusbar.push(self.statusbar_cid, 'Streaming gestoppt')

    def disable_darkice(self):
        self.darkice_start_menu_item.set_sensitive(False)
        self.darkice_togglebutton.set_sensitive(False)      

    def change_darkice_togglebutton_color(self, is_active):
        if is_active:
            self.darkice_togglebutton.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#ffe0e0"))
            #self.darkice_togglebutton.modify_bg(gtk.STATE_ENTER, gtk.gdk.color_parse("#ffe0e0"))     
            self.darkice_togglebutton.modify_bg(gtk.STATE_ACTIVE, gtk.gdk.color_parse("#978686"))
        else:
            self.darkice_togglebutton.modify_bg(gtk.STATE_NORMAL, gtk.gdk.color_parse("#e8ffe0"))
            #self.darkice_togglebutton.modify_bg(gtk.STATE_ENTER, gtk.gdk.color_parse("#e8ffe0"))     
            self.darkice_togglebutton.modify_bg(gtk.STATE_ACTIVE, gtk.gdk.color_parse("#7c8778"))

    def reset_default_status(self):        
        status = 'Recording Gui'
        
        self.statusbar.pop(self.statusbar_cid)
        self.statusbar.push(self.statusbar_cid, status)

    def is_darkice_running(self):
        if not self.darkice: return False
        return self.darkice.darkice.isalive()

    def is_ardour_running(self):
        if not self.ardour: return False
        return self.ardour.ardour.isalive()

    def start_darkice(self):
        di = streaming.Streaming()
        if not di.start():
            self.error_message(di.errors[-1])
            return False
        
        self.darkice = di
        #self.update_darkice_output()
        #GeneratorTask(self.darkice_observation_timer(), self.restart_darkice).start(5)

        return True

    def darkice_observation_timer(self):
        time.sleep(5)
        return self.is_darkice_running()

    def restart_darkice(alive):
        if not alive:
            self.error_message('darkice neu gestartet')
            #self.darkice_toggle_state(self)
  
    def update_darkice_output(self):
        tview = self.darkice_output_textview
        tbuffer = tview.get_buffer()
        try:
            while gtk.events_pending():
                gtk.main_iteration()

            line = self.darkice.darkice.readline()
            while line:
                while gtk.events_pending():
                    gtk.main_iteration()
                print line
                tbuffer.insert_at_cursor(line)
                tview.scroll_to_mark(self.textMark, 0.05, True, 0.0, 1.0)
                line = self.darkice.darkice.readline()
        except: pass

    def stop_darkice(self):
        self.darkice.darkice.close(True)
        self.darkice = None

    def killall_darkice(self):
        if not self.darkice:
            di = streaming.Streaming()
            di.killall_darkice()

    def start_ardour(self):
        ar = ardour.Ardour()
        if not ar.start():
            self.error_message(ar.errors[-1])
            return False
        
        self.ardour = ar
        
    # We use the initialization of the TutorialTextEditor class to establish
    # references to the widgets we'll need to work with in the callbacks for
    # various signals. This is done using the XML file we created with Glade    
    def __init__(self):
        
        # Default values
        self.about_dialog = None
        self.ardour = None
        self.darkice = None

        # use GtkBuilder to build our interface from the XML file 
        try:
            builder = gtk.Builder()
            builder.add_from_file(os.path.dirname(__file__)+"/recording.xml") 
        except:
            self.error_message("Failed to load UI XML file: "+os.path.dirname(__file__)+"/recording.xml")
            sys.exit(1)
            
        # get the widgets which will be referenced in callbacks
        self.window = builder.get_object("window")
        self.statusbar = builder.get_object("statusbar")
        self.darkice_output_textview = builder.get_object("darkice_output_textview")
        self.darkice_togglebutton = builder.get_object("darkice_togglebutton")
        self.darkice_start_menu_item = builder.get_object("darkice_start_menu_item")
        self.darkice_stop_menu_item = builder.get_object("darkice_stop_menu_item")
        self.ardour_button = builder.get_object("ardour_button")
        self.ardour_start_menu_item = builder.get_object("ardour_start_menu_item")
        
        # connect signals
        builder.connect_signals(self)
        
        # perpare text view (font and textmark)
        tview = self.darkice_output_textview
        tbuffer = tview.get_buffer()
        tview.modify_font(pango.FontDescription("monospace 10"))
        self.textMark = tbuffer.create_mark(None, tbuffer.get_iter_at_offset(0))

        # set color of streaming button
        #self.change_darkice_togglebutton_color(False)

        
        # set the default icon to the GTK "edit" icon
        gtk.window_set_default_icon_name(gtk.STOCK_MEDIA_RECORD)
        
        # setup and initialize our statusbar
        self.statusbar_cid = self.statusbar.get_context_id("Recording Gui")
        self.reset_default_status()

    # Run main application window
    def main(self):
        self.window.show()
        gtk.main()
    
if __name__ == "__main__":
    app = RecordingGui()
    app.main()
    
